////////////////////////////////////
#include <project.h>
#include <stdio.h>
#include <stdlib.h>
#include <cypins.h>

#include "main.h"

////////////////////////////////////
int g_echo;

////////////////////////////////////
void (*line_break_func)(void);


////////////////////////////////////
int main()
{
    CyGlobalIntEnable; /* Enable global interrupts. */
    
	// init ////////////
	g_echo          = 0;
	line_break_func = line_break_NONE;
	
	init_gpio();
	UART_1_Start();
	
	int  led = 0;
	int  bt  = 0;
	int  wd  = 0;

	user_led_Write(1);

	for(;;){
		
		bt = wd = 0;
		
		switch(UART_getc()){
			case 'l': // set low byte
				bt = getbyte();
				if(bt < 0) break;
				IO_low_Write(bt);
				break;
				
			case 'h': // set high byte
				bt = getbyte();
				if(bt < 0) break;
				IO_high_Write(bt);
				break;
				
			case 'w': // set word
				wd = getword();
				if(wd < 0) break;
				IO_low_Write(wd & 0xFF);
				IO_high_Write((wd >> 8) & 0xFF);
				break;
				
			case 'r':
				bt = IO_low_Read();
				put_hex(bt);
				line_break_func();
				break;
				
			case 'R':
				bt = IO_high_Read();
				put_hex(bt);
				line_break_func();
				break;
				
			case 'D':
				bt = IO_high_Read();
				put_hex(bt);
				bt = IO_low_Read();
				put_hex(bt);
				line_break_func();
				break;
				
			case 'c':
				IO_low_Write(0);
				break;
			
			case 'F': // clear low and high byte
				IO_low_Write(0);
				// fallthrough
				
			case 'C':
				IO_high_Write(0);
				break;
				
			case '*':
				bt = config();
				break;
				
			case '.':
				bt = -1;
				break;
				
			case '?':
				usage();
				break;
		}
		
		if(bt < 0 || wd < 0) UART_1_UartPutChar('#');

		user_led_Write(led);
		led = 1 - led;
	}
	
	return 0; // never reached
}


////////////////////////////////////
int config()
{
	int wd = 0;
	int bt = 0;
	int ack = 0;
	
	switch(UART_getc()){
		case 'D': // set pin direction (1=in, 0=out; default FFFF)
			wd = getword();
			if(wd < 0) return -1;
			set_drive_mode(wd);
			ack = 1;
			break;
			
		case 'P': // ping
			UART_1_UartPutString("!PONG");
			break;
			
		case 'V': // get FW version; answer: "!<MMmm>."
			UART_1_UartPutString("!" GPIO232_16_VERSION);
			break;
			
		case 'Y':
			bt = getbyte();
			if(bt == 0x42) CySoftwareReset();
			return -1;
			
		case 'X': // system function
			bt = getbyte();
			if(bt < 0) return -1;
			
			switch(bt & 0xF0){
				case 0x00: // echo
					g_echo = bt & 0x0F;
					break;
				
				case 0x10: // line break function
					switch(bt & 0x0F){
						case 1: line_break_func = line_break_LF;   break;
						case 2: line_break_func = line_break_CR;   break;
						case 3: line_break_func = line_break_CRLF; break;
						default: line_break_func = line_break_NONE;
					}
					break;
				
			}
			
			ack = 1;
			break;
			
		case '.':
			return -1;
	}
	
	if(ack) UART_1_UartPutChar('!');
	
	return 0;
}


////////////////////////////////////
void init_gpio()
{
	set_drive_mode(0xFFFF);
}

////////////////////////////////////
void line_break_NONE(){}
void line_break_CR(){ UART_1_UartPutChar('\r'); }
void line_break_LF(){ UART_1_UartPutChar('\n'); }
////////////////////////////////////
void line_break_CRLF(){
	// probably faster than sending a string: "\r\n"
	UART_1_UartPutChar('\r');
	UART_1_UartPutChar('\n');
}

////////////////////////////////////
void put_hex(uint8 value)
{
	UART_1_UartPutChar(HEX_DIGITS[(value >> 4) & 0x0F]);
	UART_1_UartPutChar(HEX_DIGITS[value & 0x0F]);
}

////////////////////////////////////
void set_drive_mode_byte(uint32 reg, uint16* mode_mask)
{
	uint32 tmp = *(reg32*)reg; // read register once
	
	int i = 8;
	while(i--){
		// go through each bit for each pin
		if((*mode_mask & 1) == 0){
			// do operation on temp variable instead of actual port register
			// output
			CY_SYS_PINS_SET_DRIVE_MODE(&tmp, 7-i, CY_SYS_PINS_DM_STRONG);
		}else{
			// input
			CY_SYS_PINS_SET_DRIVE_MODE(&tmp, 7-i, CY_SYS_PINS_DM_DIG_HIZ);
		}
		*mode_mask >>= 1;
	}
	
	*(reg32*)reg = tmp; // write port register once
}

////////////////////////////////////
void set_drive_mode(uint16 mode_mask)
{
	set_drive_mode_byte(IO_low__PC, &mode_mask);
	// mode_mask is shifted by 1st call of set_drive_mode_byte()
	// so no need to extract the MSB
	set_drive_mode_byte(IO_high__PC, &mode_mask);
}

// blocking function
char UART_getc()
{
	char ch;
	while((ch = UART_1_UartGetChar()) == 0);
	if(g_echo) UART_1_UartPutChar(ch);
	return ch;
}


////////////////////////////////////
int getbyte()
{
	int val = 0;
	int i = 0;
	for(; i<2; i++){
		uint8 ch = UART_getc();
		if(ch == '.') return -1;
		if(ch > '9')
			ch -= 'a' - 10;
		else
			ch -= '0';
		val = (val << 4) + ch;
	}
	return val;
}

////////////////////////////////////
int getword()
{
	int val = 0;
	int i = 0;
	for(; i<4; i++){
		uint8 ch = UART_getc();
		if(ch == '.') return -1;
		if(ch > '9')
			ch -= 'a' - 10;
		else
			ch -= '0';
		val = (val << 4) + ch;
	}
	return val;
}

////////////////////////////////////
void usage()
{
#define _PS UART_1_UartPutString
	
	_PS("\r\n== GPIO232-16 | by pleXus (2018) | " GPIO232_16_VERSION " ==\r\n");
	_PS("<nn>   = one HEX byte (lower case letters !!)\r\n");
	_PS("<nnnn> = two HEX bytes (one HEX word)\r\n");
	_PS("------------------------------\r\n");
	_PS("l<nn>    - set low byte\r\n");
	_PS("h<nn>    - set high byte\r\n");
	_PS("w<nnnn>  - set word\r\n");
	_PS("c        - clear low byte\r\n");
	_PS("C        - clear high byte\r\n");
	_PS("F        - clear word (flush)\r\n");
	_PS("r        - read low byte; returns 2 HEX digits \"hh\"\r\n");
	_PS("R        - read high byte\r\n");
	_PS("D        - read word (dump); returns 4 HEX digits \"hhhh\" (MSB left)\r\n");
	_PS(".        - cancel transmission, acknowledged by '#'\r\n");
	_PS("?        - this help page\r\n");
	_PS("------------------------------\r\n");
	_PS("*D<nnnn> - set pin direction (1=in, 0=out; default FFFF)\r\n");
	_PS("*V       - get FW version; answer: \"!<version>\"\r\n");
	_PS("*P       - ping; answer: \"!PONG\"\r\n");
	_PS("*Y42     - reset device\r\n");
	_PS("------------------------------\r\n");
	_PS("*X0<x>   - console echo; x==0 -> off; x!=0 -> on\r\n");
	_PS("*X1<x>   - append line break after read\r\n");
	_PS("           x-value: 0 = off, 1 = '\\n', 2 = '\\r', 3 = '\\r\\n'\r\n");

}


/* [] END OF FILE */
